module srisc8(input clock, input reset, 
	output [11:0] pcx, output [31:0] ir, output [31:0] results,
	output [11:0] address, output mem_op);


reg [7:0] pc;
wire [7:0] newpc;
// make program counter look like byte-addressed memory
assign pcx = {2'b00,pc,2'b00};
assign address = {2'b00,mem_address,2'b00};

reg [1:0] state;
parameter
    reset_pc = 2'd0,
    decode   = 2'd1,
    exec = 2'd2,
    halt = 2'd3;


// finite state machine
always @(posedge clock, negedge reset)
    begin
    if (~reset) state<=reset_pc;
    else
    case (state)
    reset_pc :
        begin
        pc <= 8'h0b;
        //data_register <= 16'h0000;
        state <= decode;
        end
    // decode opcode, increment program counter
    decode :
        begin
		if (mem_op) state <= exec;
		else 
			begin
			state <= decode;
			pc <= newpc;
			end
        end
    // execute instruction, setup instruction address
    exec :
        begin
        //data <= q;
        state <= decode;
		pc <= newpc;
        end
    halt :
        state <= halt;
    // default state transition
    default : state <= decode;
    endcase
    end

// instantiate memory
reg [7:0] mem_address;
wire [31:0] q;
mem  cpu_memory(
	.address(mem_address),
	.byteena(4'hf),
	.clock(clock),
	.data(db),
	.wren(MemWrite),
	.q(q));

always
	case (state)
	decode:  mem_address <= mem_op? dc1[9:2]: newpc;
	exec: mem_address <= newpc;
	default: mem_address <= 8'h0b;
	endcase


// instantiate register file

wire [31:0] da, db, dc;
wire [4:0] ra, rb, rc;

reg_file regs(
	.clock(clock),
	.reset(reset),
	.da(da),
	.db(db),
	.dc(dc),
	.ra(ra),
	.rb(rb),
	.rc(rc),
	.write_enable(RegWrite)
	);

// decode instruction

assign ir = q;
wire [5:0] op, opx;
wire [31:0] immedse;
wire [4:0] shf;
assign op = ir[5:0];
assign ra = ir[31:27];
assign rb = ir[26:22];
assign opx = ir[16:11];
assign shf = ir[10:6];
assign immedse = {(ir[21]?16'hffff:16'h0000),ir[21:6]};

wire iflag;
assign iflag = (op!=6'h3a);
assign rc = (call? 5'h1f: (iflag? rb: ir[21:17]));

//--------------------------------------------------------
// instantiate r-format control unit
wire [5:0] rcontrol;
rformat unit1(
	.opx(opx),
	.control(rcontrol)
	);
	
// instantiate i-format control unit
wire [5:0] icontrol;
wire [1:0] muxc;
wire ibranch;
iformat unit2(
	.op(op),
	.control(icontrol),
	.muxc(muxc),
	.branch(ibranch)
	);

wire [5:0] control = (iflag? icontrol: rcontrol);

//----------------------------------------------------------
// instantiate input multiplexer

wire [31:0] dmuxi; // muxi output
wire [1:0] muxi_control  = (iflag? muxc: 2'b00);
muxi mux1(
	.dst(dmuxi),
	.b(db),
	.immedse(immedse),
	.control(muxi_control)
	);
//----------------------------------------------------------
// instantiate ALU

wire [31:0] dc1; // alu output
wire [2:0] cmp_control = {1'b0,control[2],1'b1};
wire do_control = (control[5:3]==3'b001);
wire [2:0] alu_control = (do_control? cmp_control: control[2:0]);

wire zero, neg, ovfl;

alu my_alu(
	.a(da),
	.b(dmuxi),
	.control(alu_control),
	.c(dc1),
	.zero(zero),
	.neg(neg),
	.ovfl(ovfl)
	);
//----------------------------------------------------------
// instantiate barrel shifter
wire [4:0] shift = (control[3]? shf: db[4:0]);
wire [31:0] dc2;

barrel brl(
	.inp(da),
	.control(control[2:0]),
	.shift(shift),
	.out(dc2)
	);
	
//----------------------------------------------------------
// include logic for compare operations
wire cmp = (control[1]? neg: zero)^control[0];


//---------------------------------------------------------
// branch logic


wire[7:0] nextpc = pc + 1'b1;
assign branch = (iflag? ibranch: 1'b0);

assign do_branch = branch&&cmp;
wire[7:0] branchpc = nextpc+(do_branch?immedse[9:2]:8'h00);

//----------------------------------------------------------
// call/jump logic

// add (|ir) so we don't allow zero-valued instruction to be interpreted as a "call"
assign call = (op==6'h00) && (|ir);
wire jmpr = (muxd==3'h7);
wire [7:0] rpc = (call? ir[13:6]:da[9:2]);
assign newpc = (call||jmpr)? rpc: branchpc;

//----------------------------------------------------------
// instantiate dc-bus multiplexer
wire [2:0] muxd = control[5:3];

muxc mux2(
	.oal(dc1),
	.obs(dc2),
	.cmp(cmp),
	.nextpc(nextpc),
	.data(q),
	.control(muxd),
	.out(dc)
	);

//---------------------------------------------------------------------
// Load/Store
wire Lw = muxd==3'b100;
wire Sw = muxd==3'b101;
assign mem_op = Lw || Sw;

wire RegWrite = !(branch||Sw);
wire MemWrite = (state==decode) && Sw;


assign results = dc; // also send output outside the module for display

endmodule
