/* \file mpipe.cpp
 *
 *  RISC Pipeline Simulator
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream>
#include <string>
using namespace std;
#include "strtoken.h"
#include "mips.h"
#include "opcodes.h"
#include "drisc.h"

void simulate(FILE *out);
void pipeline(FILE *out);


int main(int argc, char *argv[])
{
	FILE *in, *out;
	string filename;
	errno_t nerr;
	if (argc<2) {
		cout << "usage: pipe16 hexfile [output file]\n";
		return EXIT_FAILURE;
	}
	filename = argv[1];
	nerr = fopen_s(&in,filename.c_str(),"r");
	if (nerr) {
		cout << "file: " << filename << " not found\n";
		return EXIT_FAILURE;
	}
	if (argc>2) {
		filename = argv[2];
		nerr = fopen_s(&out,filename.c_str(),"wt");
		if (nerr) {
			cout << "file: " << filename << " could not be opened\n";
			return EXIT_FAILURE;
		}
	}
	else out = stdout;
	
	// single cycle simulation
	read_memory(in);
	fprintf(out,"<html>\n");
	fprintf(out,"<h3>Source Code</h3>\n");
	fprintf(out,"<p><pre>\n");
	fprintf(out,"</pre>\n");
	simulate(out);
	//fprintf(out,"<h3>Registers</h3>\n");
	//show_registers(out);
	/*
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	*/
	
	// pipeline
	
	rewind(in);
	read_memory(in);
	fclose(in);
#define __PIPE
	pipeline(out);
	fprintf(out,"<h3>Registers</h3>\n");
	show_registers(out);
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	
	return EXIT_SUCCESS;
}

void showpc(FILE *out, int pc, bool color)
{
	if (color) fprintf(out,"<td bgcolor=E0E0E0");
	else fprintf(out,"<td");
	fprintf(out," >%04x",pc);
}

#if defined(__PIPE)

class iface {
public:
	bool active;
	bool color;
	bool halt;
	int ncycle;
	int instr;
	int pc;
	void init();
};

void iface::init()
{
	active = false;
	instr = 0;
	pc = 0;
}

class IFID: public iface {
public:
	int op, fn;
	int rd, ra, rb;
	void debug(FILE *out);
	void show(FILE *out);
	void bubble();
};

void IFID::debug(FILE *out)
{
	char *str;
	fprintf(out,"ncycle %d pc 0x%x\t",ncycle+1,pc);
	if (instr==0) str = "nop";
	else str = memnomic(op,fn);
	fprintf(out,"%s\t",halt?"halt":str);
	fprintf(out,"IFID\trd %d ra %d rb %d\n",rd,ra,rb);
}


void IFID::show(FILE *out)
{
	char *str;
	if (active) {
		showpc(out,pc,color);
		if (instr==0) str = "nop";
		else str = memnomic(op,fn);
		fprintf(out,"<br>%s",halt? "halt":str);
	}
	else fprintf(out,"<TD><BR>");
}

void IFID::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	op = 0;
	fn = 0;
	rd = 0;
}

class IDEX: public iface {
public:
	int control, muxc;
	int rd, ra, rb;
	int qa, qb;
	int bcontrol;
	int immedse;
	bool jump;
	int newpc;
	void debug(FILE *out);
	void show(FILE *out);
	void bubble();
	void setreg();
};

void IDEX::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"IDEX\trd %d ra %d rb %d qa 0x%x qb 0x%x\n",rd,ra,rb,qa,qb);
}


void IDEX::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	control = 0;
	muxc = 0;
	rd = 0;
	ra = 0;
	rb = 0;
}

void IDEX::setreg()
{
	qa = reg(ra);
	qb = reg(rb);
}

void IDEX::show(FILE *out)
{
	mips_format_t ir;
	ir.w = instr;
	int op = ir.op;
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>\n");
		else if (bcontrol>0){
			fprintf(out,"<br>%d %d 0x%x\n",ra,rb,immedse&0xFFFF);
			fprintf(out,"<br>qa %d qb %d\n",qa,qb);
		}
		else if (op>1) fprintf(out,"<br>%d %d 0x%x\n",rd,ra,immedse&0xFFFF);
		else fprintf(out,"<br>%d %d %d\n",rd,ra,rb);
	}
	else fprintf(out,"<td><br>");
}

class EXMEM: public iface {
public:
	int mc;
	bool jump;
	int rd;
	int qb;
	int qc;
	bool bubble;
	int qd;
	void show(FILE *out);
	void debug(FILE *out);
};

void EXMEM::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"EXMEM\trd %d mc %d data 0x%x address 0x%x",rd,mc,qb,qc);
	if (bubble) fprintf(out,"\tbubble qd 0x%x",qd);
	fprintf(out,"\n");
}


void EXMEM::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>\n");
		else fprintf(out,"<br> 0x%x\n",qc);
	}
	else fprintf(out,"<td><br>");
}

class MEMWB: public iface {
public:
	bool reg_write_enable;
	int mc;
	int maddr;
	int rd;
	int qd;
	void debug(FILE *out);
	void show(FILE *out);
	void showmem(FILE *out);
};

void MEMWB::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"MEMWB\trd %d qd 0x%x\n",rd,qd);
}

void MEMWB::showmem(FILE *out)
{
	if (mc==LW_C) fprintf(out,"qd = 0x%x =  mem[0x%x]\n",qd,maddr);
	if (mc==SW_C) fprintf(out,"mem[0x%x] = 0x%x\n",maddr,qd);
}

void MEMWB::show(FILE *out)
{
	bool flag = (mc==LW_C || mc==SW_C);
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>");
		else if (flag) fprintf(out,"<br>0x%x",qd);
		else fprintf(out,"<br><br>");
	}
	else fprintf(out,"<td><br><br>");
}

class WBIF: public iface {
public:
	int rd;
	int qd;
	void show(FILE *out);
	bool show(FILE *out, int pcx, int qdx);
};

void WBIF::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		fprintf(out,"<br>0x%x",qd);
	}
	else fprintf(out,"<td><br>");
}

bool WBIF::show(FILE *out,int pcx,int qdx)
{
	bool flag = qd==qdx && pc==pcx;
	if (flag) fprintf(out,"<td %s",color? "bgcolor=\"E0E0E0\">":">");
	else fprintf(out,"<td bgcolor=FF0000>");
	fprintf(out,"%04x<br>0x%x\n",pcx,qdx);
	return !flag;
}



IFID IFID_old, IFID_new;
IDEX IDEX_old, IDEX_new;
EXMEM EXMEM_old, EXMEM_new;
MEMWB MEMWB_old, MEMWB_new;
WBIF WBIF_old, WBIF_new;

#define NO_FORWARDING

void stage_IF(int nextpc, int ncycle, IFID &IFID_old, IFID &IFID_new) {
	mips_format_t ir;
	int op, fn;
	int ra, rb, rd;
	int range_error;
	// IF stage
	range_error = inst_memory(nextpc,ir.w);
	IFID_new.active = !IFID_old.halt;
	IFID_new.color = (ncycle&3)==2;
	IFID_new.ncycle = ncycle;
	IFID_new.instr = ir.w;
	IFID_new.halt = (ir.w==HALT) || IFID_old.halt;
	IFID_new.pc = nextpc;
	// obtain instruction fields
	op = ir.op;
	rd = ir.rd;
	ra = ir.rs;
	rb = ir.rt;
	fn = ir.fn;
	if (op>1) rd = rb;
	else if (op==1) {
		fn = 64+rb;
		rd = 0;
		rb = 0;
	}
	IFID_new.op = op;
	IFID_new.fn = fn;
	bool flag = op==SW;
	IFID_new.rd = flag? 0: rd;
	if (IFID_new.halt) {
		IFID_new.ra = 0;
		IFID_new.rb = 0;
	}
	else {
		IFID_new.ra = ra;
		IFID_new.rb = rb;
	}
}

void copy(iface &iface_old, iface &iface_new)
{
	iface_new.active = iface_old.active;
	iface_new.color = iface_old.color;
	iface_new.halt = iface_old.halt;
	iface_new.ncycle = iface_old.ncycle;
	iface_new.instr = iface_old.instr;
	iface_new.pc = iface_old.pc;
}	

void stage_ID(IFID &IFID_old, IDEX &IDEX_new)
{
	mips_format_t ir;
	int op, fn;
	int bcontrol, rcontrol, icontrol, control;
	int imux, muxc;
	ir.w = IFID_old.instr;
	op = IFID_old.op;
	fn = IFID_old.fn;
	rformat(fn,rcontrol);
	iformat(op,icontrol,imux);
	control = op==0? rcontrol: icontrol;
	muxc = op==0? 0: imux;
	int immed = ir.immed;
	int immedse = immed;
	if (immed>>15) immedse = 0xFFFF0000| immed&0xFFFF;
	bformat(op,fn,bcontrol);

	/*
	IDEX_new.active = IFID_old.active;
	IDEX_new.color = IFID_old.color;
	IDEX_new.ncycle = IFID_old.ncycle;
	IDEX_new.instr = IFID_old.instr;
	IDEX_new.halt = IFID_old.halt;
	IDEX_new.pc = IFID_old.pc;
	*/
	copy(IFID_old,IDEX_new);
	IDEX_new.control = control;
	IDEX_new.muxc = muxc;
	IDEX_new.rd = (bcontrol>0? 0: IFID_old.rd);
	IDEX_new.ra = IFID_old.ra;
	IDEX_new.rb = IFID_old.rb;
	IDEX_new.bcontrol = bcontrol;
	IDEX_new.immedse = immedse;
	IDEX_new.setreg();
	/*
	if (bcontrol>0) {
		int cmp;
		compare(IDEX_new.qa,IDEX_new.qb,bcontrol,cmp);
		IDEX_new.jump = cmp!=0;
	}
	else IDEX_new.jump = false;
	*/
}


void branch_control(IDEX &IDEX_new, EXMEM &EXMEM_new, FILE *out)
{
	bool flag;
	int cmp;
	IDEX_new.jump = false;
	if (IDEX_new.bcontrol==0) return;
	flag = (IDEX_new.ra==EXMEM_new.rd && IDEX_new.ra!=0); 
	int qa = (flag? EXMEM_new.qc: IDEX_new.qa);
	if (flag) {
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFCCFF colspan=6>");
		fprintf(out,"branch control: qa 0x%x (ra: %d)\n",qa,IDEX_new.ra);
	}
	flag = (IDEX_new.rb==EXMEM_new.rd && IDEX_new.rb!=0); 
	int qb = (flag? EXMEM_new.qc: IDEX_new.qb);
	if (flag) {
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFCCFF colspan=6>");
		fprintf(out,"branch control: qb 0x%x (rb: %d)\n",qb,IDEX_new.rb);
	}
	compare(qa,qb,IDEX_new.bcontrol,cmp);
	IDEX_new.jump = cmp!=0;
}

void ex_detail(int pc, int qa, int qb, int qt, int qc)
{
	printf("*** pc 0x%x qa 0x%x qb 0x%x qt 0x%x qc 0x%x\n",pc,qa,qb,qt,qc);
}

void stage_EX(IDEX &IDEX_old, EXMEM &EXMEM_new)
{
	mips_format_t ir;
	EXMEM_new.active = IDEX_old.active;
	EXMEM_new.color = IDEX_old.color;
	EXMEM_new.halt = IDEX_old.halt;
	EXMEM_new.ncycle = IDEX_old.ncycle;
	EXMEM_new.pc = IDEX_old.pc;
	EXMEM_new.instr = IDEX_old.instr;
	if (!IDEX_old.active) return;
	EXMEM_new.rd = IDEX_old.rd;
	ir.w = IDEX_old.instr;
	int qa = IDEX_old.qa;
	int qb = IDEX_old.qb;
	int control = IDEX_old.control;
	int muxc = IDEX_old.muxc;
	//int jump = 0;
	int qt = mux1(qb, ir.immed, muxc);
	int qc = alu(control&0xF, qa, qt);
	/*
	int pc = IDEX_old.pc;
	if (pc==6) ex_detail(pc,qa,qb,qt,qc);
	*/
	//
	EXMEM_new.mc = (control>>4)&0xF;
	EXMEM_new.qb = qb;
	EXMEM_new.qc = (IDEX_old.bcontrol>0? 0: qc);
	EXMEM_new.bubble = false;
}

void mem_detail(int pc, int mc, int qb, int qc, int qm)
{
	printf("*** pc 0x%x mc %d  qb 0x%x qc 0x%x qm 0x%x\n",pc,mc,qb,qc,qm);
}

void stage_MEM(EXMEM &EXMEM_old, MEMWB &MEMWB_new)
{
	copy(EXMEM_old,MEMWB_new);
	MEMWB_new.rd = EXMEM_old.rd;
	int mc = EXMEM_old.mc;
	MEMWB_new.mc = mc;
	bool mem_access = mc==LW_C || mc==SW_C;
	int qc = EXMEM_old.qc;
	int maddr = mem_access? qc: 0;
	int qb = EXMEM_old.qb;
	int qm;
	int range_error =  data_memory(maddr, qb, mc==SW_C, qm);
	if (range_error) printf("range error addr %x\n",maddr);
	int qd = (mc==LW_C? qm: (mc==SW_C? qb: qc));
	/* debug
	int pc = EXMEM_old.pc;
	if (pc==5) mem_detail(pc,mc,qb,qc,qm);
	*/
	MEMWB_new.maddr = maddr;
	MEMWB_new.qd = qd;
	//printf("MEMWB 0x%x mc %d qd %d\n",MEMWB_new.pc,mc,qd);
	if (EXMEM_old.bubble) MEMWB_new.qd = qm;
}

void stage_WB(MEMWB &MEMWB_old, WBIF &WBIF_new)
{
	/*
	WBIF_new.active = MEMWB_old.active;
	WBIF_new.color = MEMWB_old.color;
	WBIF_new.halt = MEMWB_old.halt;
	WBIF_new.ncycle = EXMEM_old.ncycle;
	WBIF_new.pc = EXMEM_old.pc;
	WBIF_new.instr = EXMEM_old.instr;
	*/
	copy (MEMWB_old,WBIF_new);
	int rd = MEMWB_old.rd;
	int qd = MEMWB_old.qd;
	reg_wb(rd,qd);
	WBIF_new.rd = rd;
	WBIF_new.qd = qd;
};

void bubble_needed(FILE *out, char *str, int rd)
{
	if (rd==0) return;
	if (!out) return;
	fprintf(out,"<tr align=center><td><br>");
	fprintf(out,"<td colspan=6 align=center>");
	fprintf(out,"bubble needed %s = %d</tr>\n",str,rd);
}

//#define DEBUG

void forward1(IDEX &IDEX_old, EXMEM &EXMEM_old, FILE *out)
{
	//int ncycle = IDEX_old.ncycle + 2;
	// yellow coding
	if (IDEX_old.ra == EXMEM_old.rd && IDEX_old.ra !=0) {
		int ra = IDEX_old.ra;
		int qa = EXMEM_old.bubble? EXMEM_old.qd: EXMEM_old.qc;
		IDEX_old.qa = qa;
		//fprintf(stderr,"(cycle %d) qa forwarded: 0x%x (ra: %d)\n",ncycle,qa,ra);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=6>qa forwarded: 0x%x (ra: %d)\n",qa,ra);
	}
	if (IDEX_old.rb == EXMEM_old.rd && IDEX_old.rb !=0) {
		int rb = IDEX_old.rb;
		int qb = EXMEM_old.bubble? EXMEM_old.qd: EXMEM_old.qc;
		IDEX_old.qb = qb;
		//fprintf(stderr,"(cycle %d) qb forwarded : 0x%x (rb: %d)\n",ncycle,qb,rb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=6>qb forwarded: 0x%x (rb: %d)\n",qb,rb);
	}
}

void forward2(IDEX &IDEX_old, MEMWB &MEMWB_old, FILE *out)
{
	//int ncycle = IDEX_old.ncycle+2;
	// light blue coding
	if (IDEX_old.ra == MEMWB_old.rd && IDEX_old.ra !=0) {
		int ra = IDEX_old.ra;
		int qa = MEMWB_old.qd;
		IDEX_old.qa = qa;
		//fprintf(stderr,"(cycle %d) qa forwarded : 0x%x (ra: %d)\n",ncycle,qa,ra);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=CCCCFF colspan=6>qa forwarded: 0x%x (ra: %d)\n",qa,ra);
	}
	if (IDEX_old.rb == MEMWB_old.rd && IDEX_old.rb !=0) {
		int rb = IDEX_old.rb;
		int qb = MEMWB_old.qd;
		IDEX_old.qb = qb;
		//fprintf(stderr,"(cycle %d) qb forwarded : 0x%x (rb: %d)\n",ncycle,qb,rb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=CCCCFF colspan=6>qb forwarded: 0x%x (rb: %d)\n",qb,rb);
	}
}

void forward3(IDEX &IDEX_new, MEMWB &MEMWB_old, FILE *out)
{
	//int ncycle = IDEX_new.ncycle + 2;
	// pink coding
	if (IDEX_new.ra == MEMWB_old.rd && IDEX_new.ra !=0) {
		int ra = IDEX_new.ra;
		int qa = MEMWB_old.qd;
		IDEX_new.qa = qa;
		//fprintf(stderr,"(cycle %d) qa transferred: 0x%x (ra: %d)\n",ncycle,qa,ra);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFCCCC colspan=6>qa transferred: 0x%x (ra: %d)\n",qa,ra);
	}
	if (IDEX_new.rb == MEMWB_old.rd && IDEX_new.rb !=0) {
		int rb = IDEX_new.rb;
		int qb = MEMWB_old.qd;
		IDEX_new.qb = qb;
		//fprintf(stderr,"(cycle %d) qb transferred : 0x%x (rb: %d)\n",ncycle,qb,rb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFCCCC colspan=6>qb transferred: 0x%x (rb: %d)\n",qb,rb);
	}
}

void pipeline(FILE *out)
{
	int nextpc, ncycle=0;
	int ndx = 0;
	bool done = false;
	int errors = 0;
	
	IFID_old.init();
	IDEX_old.init();
	EXMEM_old.init();
	MEMWB_old.init();
	WBIF_old.init();

	if (out) {
		fprintf(out,"<h3>Pipeline Results</h3>\n");
		fprintf(out,"<p><table border>\n");
		fprintf(out,"<tr bgcolor=CCCCFF><th>#<th>IF<th>ID<th>EX<th>MEM<th>WB<th>ok\n");
	}
	nextpc = 0;
	ncycle = 0;
	bool bubble = false;
	while (!done) {
		forward2(IDEX_old,MEMWB_old,out);
		forward1(IDEX_old,EXMEM_old,out);
		// IF stage
		stage_IF(nextpc,ncycle,IFID_old,IFID_new);
		// ID stage
		stage_ID(IFID_old,IDEX_new);
		forward3(IDEX_new,MEMWB_old,out);
		//  EX Stage
		stage_EX(IDEX_old,EXMEM_new);
		branch_control(IDEX_new,EXMEM_new,out);
		// MEM Stage
		stage_MEM(EXMEM_old,MEMWB_new);
		// WB Stage
		stage_WB(MEMWB_old,WBIF_new);
		//
		if (++ncycle>=MAX_CYCLES) break;
		if (WBIF_new.instr==HALT) break;
		
		// tell us if a memory hazard bubble is needed
		if (EXMEM_new.mc==LW_C && !bubble) {
			int rd = EXMEM_new.rd;
			bubble = true;
			// change out to 0 to suppress output
			if (IDEX_new.ra==rd) bubble_needed(0,"ra",rd);
			else if (IDEX_new.rb==rd) bubble_needed(0,"rb",rd);
			else bubble = false;
		}
		else bubble = false;
		if (bubble) {
			IFID_new = IFID_old;
			IDEX_new.bubble();
			EXMEM_new.bubble = true;
			//EXMEM_new.mc = 0;
			EXMEM_new.qd = MEMWB_old.qd;
		}
		if (IDEX_old.jump) {
			IDEX_new.bubble();
			EXMEM_new.bubble = true;
			//EXMEM_new.mc = 0;
			EXMEM_new.qd = MEMWB_old.qd;
		}
		// show state
		fprintf(out,"<tr align=center><td>%d",ncycle);
		IFID_new.show(out);
		IDEX_new.show(out);
		EXMEM_new.show(out);
		MEMWB_new.show(out);
		WBIF_new.show(out);
		if (WBIF_new.active) {
			int qd = steps[ndx].qd;
			int pc = steps[ndx].pc;
			if (WBIF_new.show(out,pc,qd)) errors++;
			ndx++;
		}
		else fprintf(out,"<td><br>\n");
		if (MEMWB_new.active && MEMWB_new.mc) {
			fprintf(out,"<tr align=center><td><br>");
			fprintf(out,"<td colspan=6 align=center>");
			//fprintf(out,"mc %d   ",MEMWB_new.mc);
			MEMWB_new.showmem(out);
		}
		// change state
		IFID_old = IFID_new;
		IDEX_old = IDEX_new;
		EXMEM_old = EXMEM_new;
		MEMWB_old = MEMWB_new;
		WBIF_old = WBIF_new;
		if (IDEX_old.jump) {
			nextpc = IDEX_old.pc + IDEX_old.immedse;
		}
		else if (!bubble) nextpc++;
	}
	if (out) fprintf(out,"</table>\n");
	if (errors>0) {
		fprintf(out,"<p>errors: %d\n",errors);
		printf("errors: %d\n",errors);
	}
}

#endif
