/* \file mpipe.cpp
 *
 *  RISC Pipeline Simulator
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream>
#include <string>
using namespace std;
#include "strtoken.h"
#include "mips.h"
#include "opcodes.h"
#include "drisc.h"

void simulate(FILE *out);
void pipeline(FILE *out);


int main(int argc, char *argv[])
{
	FILE *in, *out;
	string filename;
	errno_t nerr;
	if (argc<2) {
		cout << "usage: pipe16 hexfile [output file]\n";
		return EXIT_FAILURE;
	}
	filename = argv[1];
	nerr = fopen_s(&in,filename.c_str(),"r");
	if (nerr) {
		cout << "file: " << filename << " not found\n";
		return EXIT_FAILURE;
	}
	if (argc>2) {
		filename = argv[2];
		nerr = fopen_s(&out,filename.c_str(),"wt");
		if (nerr) {
			cout << "file: " << filename << " could not be opened\n";
			return EXIT_FAILURE;
		}
	}
	else out = stdout;
	
	// single cycle simulation
	read_memory(in);
	fprintf(out,"<html>\n");
	simulate(out);
	//fprintf(out,"<h3>Registers</h3>\n");
	//show_registers(out);
	/*
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	*/
	
	// pipeline
	
	rewind(in);
	read_memory(in);
	fclose(in);
#define __PIPE
	pipeline(out);
	fprintf(out,"<h3>Registers</h3>\n");
	show_registers(out);
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	
	return EXIT_SUCCESS;
}

void showpc(FILE *out, int pc, bool color)
{
	if (color) fprintf(out,"<td bgcolor=E0E0E0");
	else fprintf(out,"<td");
	fprintf(out," >%04x",pc);
}

#if defined(__PIPE)

class iface {
public:
	bool active;
	bool color;
	bool halt;
	int ncycle;
	int instr;
	int pc;
	void init();
};

void iface::init()
{
	active = false;
	instr = 0;
	pc = 0;
}

class IFID: public iface {
public:
	int op, fn;
	int rd, ra, rb;
	void debug(FILE *out);
	void show(FILE *out);
	void bubble();
};

void IFID::debug(FILE *out)
{
	char *str;
	fprintf(out,"ncycle %d pc 0x%x\t",ncycle+1,pc);
	str = op==0? funcname[fn]: opname[op];
	fprintf(out,"%s\t",halt?"halt":str);
	fprintf(out,"IFID\trd %d ra %d rb %d\n",rd,ra,rb);
}


void IFID::show(FILE *out)
{
	char *str;
	if (active) {
		showpc(out,pc,color);
		str = op==0? funcname[fn]: opname[op];
		fprintf(out,"<br>%s",halt? "halt":str);
	}
	else fprintf(out,"<TD><BR>");
}

void IFID::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	op = 0;
	fn = 0;
	rd = 0;
}

class IDEX: public iface {
public:
	int control, muxc;
	int rd, ra, rb;
	int qa, qb;
	void debug(FILE *out);
	void show(FILE *out);
	void bubble();
	void setreg();
};

void IDEX::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"IDEX\trd %d ra %d rb %d qa 0x%x qb 0x%x\n",rd,ra,rb,qa,qb);
}


void IDEX::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	control = 0;
	muxc = 0;
	rd = 0;
	ra = 0;
	rb = 0;
}

void IDEX::setreg()
{
	qa = reg(ra);
	qb = reg(rb);
}

void IDEX::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>\n");
		else fprintf(out,"<br>%d %d %d\n",rd,ra,rb);
	}
	else fprintf(out,"<td><br>");
}

class EXMEM: public iface {
public:
	int mc;
	int jump;
	int rd;
	int qb;
	int qc;
	bool bubble;
	int qd;
	void show(FILE *out);
	void debug(FILE *out);
};

void EXMEM::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"EXMEM\trd %d mc %d data 0x%x address 0x%x",rd,mc,qb,qc);
	if (bubble) fprintf(out,"\tbubble qd 0x%x",qd);
	fprintf(out,"\n");
}


void EXMEM::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>\n");
		else fprintf(out,"<br> 0x%x\n",qc);
	}
	else fprintf(out,"<td><br>");
}

class MEMWB: public iface {
public:
	bool reg_write_enable;
	int mc;
	int jump;
	int maddr;
	int rd;
	int qd;
	void debug(FILE *out);
	void show(FILE *out);
	void showmem(FILE *out);
};

void MEMWB::debug(FILE *out)
{
	fprintf(out,"ncycle %d pc 0x%x\t-\t",ncycle+1,pc);
	fprintf(out,"MEMWB\trd %d qd 0x%x\n",rd,qd);
}

void MEMWB::showmem(FILE *out)
{
	if (mc==LW_C) fprintf(out,"qd = 0x%x =  mem[0x%x]\n",qd,maddr);
	if (mc==SW_C) fprintf(out,"mem[0x%x] = 0x%x\n",maddr,qd);
}

void MEMWB::show(FILE *out)
{
	bool flag = (mc==LW_C || mc==SW_C);
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>");
		else if (flag) fprintf(out,"<br>0x%x",qd);
		else fprintf(out,"<br><br>");
	}
	else fprintf(out,"<td><br><br>");
}

class WBIF: public iface {
public:
	int rd;
	int qd;
	void show(FILE *out);
	void show(FILE *out, int pcx, int qdx);
};

void WBIF::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		fprintf(out,"<br>0x%x",qd);
	}
	else fprintf(out,"<td><br>");
}

void WBIF::show(FILE *out,int pcx,int qdx)
{
	bool flag = qd==qdx && pc==pcx;
	if (flag) fprintf(out,"<td %s",color? "bgcolor=\"E0E0E0\">":">");
	else fprintf(out,"<td bgcolor=FF0000>");
	fprintf(out,"%04x<br>0x%x\n",pcx,qdx);
}



IFID IFID_old, IFID_new;
IDEX IDEX_old, IDEX_new;
EXMEM EXMEM_old, EXMEM_new;
MEMWB MEMWB_old, MEMWB_new;
WBIF WBIF_old, WBIF_new;

#define NO_FORWARDING

void stage_IF(int nextpc, int ncycle, IFID &IFID_old, IFID &IFID_new) {
	mips_format_t ir;
	int op, fn;
	int ra, rb, rd;
	int range_error;
	// IF stage
	range_error = inst_memory(nextpc,ir.w);
	IFID_new.active = !IFID_old.halt;
	IFID_new.color = (ncycle&3)==2;
	IFID_new.ncycle = ncycle;
	IFID_new.instr = ir.w;
	IFID_new.halt = (ir.w==HALT) || IFID_old.halt;
	IFID_new.pc = nextpc;
	// obtain instruction fields
	op = ir.op;
	rd = ir.rd;
	ra = ir.rs;
	rb = ir.rt;
	fn = ir.fn;
	if (op>0) rd = rb;

	IFID_new.op = op;
	IFID_new.fn = fn;
	bool flag = op==SW || op==BNE;
	IFID_new.rd = flag? 0: rd;
	if (IFID_new.halt) {
		IFID_new.ra = 0;
		IFID_new.rb = 0;
	}
	else {
		IFID_new.ra = ra;
		IFID_new.rb = rb;
	}
}

void copy(iface &iface_old, iface &iface_new)
{
	iface_new.active = iface_old.active;
	iface_new.color = iface_old.color;
	iface_new.halt = iface_old.halt;
	iface_new.ncycle = iface_old.ncycle;
	iface_new.instr = iface_old.instr;
	iface_new.pc = iface_old.pc;
}	

void stage_ID(IFID &IFID_old, IDEX &IDEX_new)
{
	int op, fn;
	int rcontrol, icontrol, control;
	int imux, muxc;
	op = IFID_old.op;
	fn = IFID_old.fn;
	rformat(fn,rcontrol);
	iformat(op,icontrol,imux);
	control = op==0? rcontrol: icontrol;
	muxc = op==0? 0: imux;

	/*
	IDEX_new.active = IFID_old.active;
	IDEX_new.color = IFID_old.color;
	IDEX_new.ncycle = IFID_old.ncycle;
	IDEX_new.instr = IFID_old.instr;
	IDEX_new.halt = IFID_old.halt;
	IDEX_new.pc = IFID_old.pc;
	*/
	copy(IFID_old,IDEX_new);
	IDEX_new.control = control;
	IDEX_new.muxc = muxc;
	IDEX_new.rd = IFID_old.rd;
	IDEX_new.ra = IFID_old.ra;
	IDEX_new.rb = IFID_old.rb;
	IDEX_new.setreg();
}

void ex_detail(int pc, int qa, int qb, int qt, int qc)
{
	printf("*** pc 0x%x qa 0x%x qb 0x%x qt 0x%x qc 0x%x\n",pc,qa,qb,qt,qc);
}

void stage_EX(IDEX &IDEX_old, EXMEM &EXMEM_new)
{
	mips_format_t ir;
	EXMEM_new.active = IDEX_old.active;
	EXMEM_new.color = IDEX_old.color;
	EXMEM_new.halt = IDEX_old.halt;
	EXMEM_new.ncycle = IDEX_old.ncycle;
	EXMEM_new.pc = IDEX_old.pc;
	EXMEM_new.instr = IDEX_old.instr;
	if (!IDEX_old.active) return;
	EXMEM_new.rd = IDEX_old.rd;
	ir.w = IDEX_old.instr;
	int qa = IDEX_old.qa;
	int qb = IDEX_old.qb;
	int control = IDEX_old.control;
	int muxc = IDEX_old.muxc;
	//int jump = 0;
	int qt = mux1(qb, ir.immed, muxc);
	int qc = alu(control&0x7, qa, qt);
	/*
	int pc = IDEX_old.pc;
	if (pc==6) ex_detail(pc,qa,qb,qt,qc);
	*/
	//
	EXMEM_new.mc = (control>>4)&0xF;
	EXMEM_new.qb = qb;
	EXMEM_new.qc = qc;
	EXMEM_new.bubble = false;
}

void mem_detail(int pc, int mc, int qb, int qc, int qm)
{
	printf("*** pc 0x%x mc %d  qb 0x%x qc 0x%x qm 0x%x\n",pc,mc,qb,qc,qm);
}

void stage_MEM(EXMEM &EXMEM_old, MEMWB &MEMWB_new)
{
	copy(EXMEM_old,MEMWB_new);
	MEMWB_new.rd = EXMEM_old.rd;
	int mc = EXMEM_old.mc;
	MEMWB_new.mc = mc;
	bool mem_access = mc==LW_C || mc==SW_C;
	int qc = EXMEM_old.qc;
	int maddr = mem_access? qc: 0;
	int qb = EXMEM_old.qb;
	int qm;
	int range_error =  data_memory(maddr, qb, mc==SW_C, qm);
	if (range_error) printf("range error addr %x\n",maddr);
	int qd = (mc==LW_C? qm: (mc==SW_C? qb: qc));
	/* debug
	int pc = EXMEM_old.pc;
	if (pc==5) mem_detail(pc,mc,qb,qc,qm);
	*/
	MEMWB_new.maddr = maddr;
	MEMWB_new.qd = qd;
	//printf("MEMWB 0x%x mc %d qd %d\n",MEMWB_new.pc,mc,qd);
	if (EXMEM_old.bubble) MEMWB_new.qd = qm;
}

void stage_WB(MEMWB &MEMWB_old, WBIF &WBIF_new)
{
	/*
	WBIF_new.active = MEMWB_old.active;
	WBIF_new.color = MEMWB_old.color;
	WBIF_new.halt = MEMWB_old.halt;
	WBIF_new.ncycle = EXMEM_old.ncycle;
	WBIF_new.pc = EXMEM_old.pc;
	WBIF_new.instr = EXMEM_old.instr;
	*/
	copy (MEMWB_old,WBIF_new);
	int rd = MEMWB_old.rd;
	int qd = MEMWB_old.qd;
	reg_wb(rd,qd);
	WBIF_new.rd = rd;
	WBIF_new.qd = qd;
};

void bubble_needed(FILE *out, char *str, int rd)
{
	if (rd==0) return;
	if (!out) return;
	fprintf(out,"<tr align=center><td><br>");
	fprintf(out,"<td colspan=6 align=center>");
	fprintf(out,"bubble needed %s = %d</tr>\n",str,rd);
}

//#define DEBUG

void forward1(IDEX &IDEX_old, EXMEM &EXMEM_old, FILE *out)
{
	int ra, rb, qa, qb;
	int ncycle = IDEX_old.ncycle + 2;
	if (IDEX_old.ra == EXMEM_old.rd && IDEX_old.ra !=0) {
		ra = IDEX_old.ra;
		qa = EXMEM_old.bubble? EXMEM_old.qd: EXMEM_old.qc;
		IDEX_old.qa = qa;
		fprintf(stderr,"(cycle %d) qa forwarded: 0x%x (ra: %d)\n",ncycle,qa,ra);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=6>qa forwarded: 0x%x (ra: %d)\n",qa,ra);
	}
	if (IDEX_old.rb == EXMEM_old.rd && IDEX_old.rb !=0) {
		rb = IDEX_old.rb;
		qb = EXMEM_old.bubble? EXMEM_old.qd: EXMEM_old.qc;
		IDEX_old.qb = qb;
		fprintf(stderr,"(cycle %d) qb forwarded : 0x%x (rb: %d)\n",ncycle,qb,rb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=6>qb forwarded: 0x%x (rb: %d)\n",qb,rb);
	}
}

void forward2(IDEX &IDEX_old, MEMWB &MEMWB_old, FILE *out)
{
	int ncycle = IDEX_old.ncycle+2;
	if (IDEX_old.ra == MEMWB_old.rd && IDEX_old.ra !=0) {
		int ra = IDEX_old.ra;
		int qa = MEMWB_old.qd;
		IDEX_old.qa = qa;
		fprintf(stderr,"(cycle %d) qa forwarded : 0x%x (ra: %d)\n",ncycle,qa,ra);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=CCCCFF colspan=6>qa forwarded: 0x%x (ra: %d)\n",qa,ra);
	}
	if (IDEX_old.rb == MEMWB_old.rd && IDEX_old.rb !=0) {
		int rb = IDEX_old.rb;
		int qb = MEMWB_old.qd;
		IDEX_old.qb = qb;
		fprintf(stderr,"(cycle %d) qb forwarded : 0x%x (rb: %d)\n",ncycle,qb,rb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=CCCCFF colspan=6>qb forwarded: 0x%x (rb: %d)\n",qb,rb);
	}
}


void pipeline(FILE *out)
{
	int nextpc, ncycle=0;
	int ndx = 0;
	bool done = false;
	
	IFID_old.init();
	IDEX_old.init();
	EXMEM_old.init();
	MEMWB_old.init();
	WBIF_old.init();

	if (out) {
		fprintf(out,"<h3>Pipeline Results</h3>\n");
		fprintf(out,"<p><table border>\n");
		fprintf(out,"<tr bgcolor=CCCCFF><th>#<th>IF<th>ID<th>EX<th>MEM<th>WB<th>ok\n");
	}
	nextpc = 0;
	ncycle = 0;
	bool bubble = false;
	while (!done) {
		forward2(IDEX_old,MEMWB_old,out);
		forward1(IDEX_old,EXMEM_old,out);
		// IF stage
		stage_IF(nextpc,ncycle,IFID_old,IFID_new);
		// ID stage
		stage_ID(IFID_old,IDEX_new);
		//  EX Stage
		stage_EX(IDEX_old,EXMEM_new);
		// MEM Stage
		stage_MEM(EXMEM_old,MEMWB_new);
		// WB Stage
		stage_WB(MEMWB_old,WBIF_new);
		//
		if (++ncycle>=MAX_CYCLES) break;
		if (WBIF_new.instr==HALT) break;
		
		// tell us if a memory hazard bubble is needed
		if (EXMEM_new.mc==LW_C && !bubble) {
			int rd = EXMEM_new.rd;
			bubble = true;
			// change out to 0 to suppress output
			if (IDEX_new.ra==rd) bubble_needed(0,"ra",rd);
			else if (IDEX_new.rb==rd) bubble_needed(0,"rb",rd);
			else bubble = false;
		}
		else bubble = false;
		if (bubble) {
			IFID_new = IFID_old;
			IDEX_new.bubble();
			EXMEM_new.bubble = true;
			//EXMEM_new.mc = 0;
			EXMEM_new.qd = MEMWB_old.qd;
		}
		// show state
		fprintf(out,"<tr align=center><td>%d",ncycle);
		IFID_new.show(out);
		IDEX_new.show(out);
		EXMEM_new.show(out);
		MEMWB_new.show(out);
		WBIF_new.show(out);
		if (WBIF_new.active) {
			int qd = steps[ndx].qd;
			int pc = steps[ndx].pc;
			WBIF_new.show(out,pc,qd);
			ndx++;
		}
		else fprintf(out,"<td><br>\n");
		if (MEMWB_new.active && MEMWB_new.mc) {
			fprintf(out,"<tr align=center><td><br>");
			fprintf(out,"<td colspan=6 align=center>");
			//fprintf(out,"mc %d   ",MEMWB_new.mc);
			MEMWB_new.showmem(out);
		}
		// change state
		IFID_old = IFID_new;
		IDEX_old = IDEX_new;
		EXMEM_old = EXMEM_new;
		MEMWB_old = MEMWB_new;
		WBIF_old = WBIF_new;
		if (!bubble) nextpc++;
	}
	if (out) fprintf(out,"</table>\n");
}
		


void parse(FILE *in, FILE *out)
{
	char buf[512];
	char *inp, *next;
	int opcode, ra, rb, rc, field;
	int count, code, n;
	string token;
	bool done = false;
	/*
	 * Go through input file, line by line
	 */
	n = 0;
	while (!done) {
		fgets(buf,511,in);
		if (feof(in)) break;
		//fprintf(out,"%s",buf);
		if (*buf=='#') continue;
		next = buf;
		/*
		 * parse current line
		 */
		count = 0;
		opcode = ra = rb = rc = 0;
		while (next) {
			/*
			 * search for '<', and copy prior text to
			 * output stream
 */
			inp = next;
			next = strqtrm(inp,',');
			sscanf_s(inp,"%d",&field);
			//fprintf(out,"count %d field %d\n",count,field);
			switch (count) {
			case 0:
				token = inp;
				//fprintf(out,"%s %d\n",token.c_str(),opcode);
				count = 1;
				break;
			case 1:
				ra = field;
				count = 2;
				break;
			case 2:
				rb = field;
				count = 3;
				break;
			case 3:
				rc = field;
				count = 4;
				break;
			default:
				break;
			}
		}
		code = ((opcode&0x7)<<13) | ((ra&0x7)<<10) | ((rb&0x7)<<7) | (rc&0x7F);;
		fprintf(out,"%04X\n",code);
		n++;
#if defined(STUFF)
			/*
			 * next points to character after ',' (if ','
			 * was found).
			 */
			/*
			if (next) {
				/*
				*  isolate (in html) text up to the character '>'
				*/
				html = next;
				next = strqtrm(html,'>');
				/*
				*  check for known codes
				*/
				string token = html;
				char *value = strqtrm(html,' ');
				if (token.substr(0,7).compare("include")==0) {
					include(out,value);
					continue; // go to next line
				}
				else if (token.substr(0,4).compare("copy")==0) {
					copy(out,value);
					continue; // go to next line
				}
				else if (token.substr(0,6).compare("insert")==0) {
					parse_insert(value,out);
				}
				/*
				 *  otherwise just echo the html code
				 */
				else fprintf(out,"<%s>",token.c_str());
			}
#endif
	}
	while (n++ < 32) fprintf(out,"0000\n");
}

#endif
