/* \file mpipe.cpp
 *
 *  RISC Pipeline Simulator
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream>
#include <string>
using namespace std;
#include "strtoken.h"
#include "mips.h"
#include "opcodes.h"
#include "drisc.h"


void simulate(FILE *out);
void pipeline(FILE *out);

int main(int argc, char *argv[])
{
	FILE *in, *out;
	string filename;
	errno_t nerr;
	if (argc<2) {
		cout << "usage: pipe16 hexfile [output file]\n";
		return EXIT_FAILURE;
	}
	filename = argv[1];
	nerr = fopen_s(&in,filename.c_str(),"r");
	if (nerr) {
		cout << "file: " << filename << " not found\n";
		return EXIT_FAILURE;
	}
	if (argc>2) {
		filename = argv[2];
		nerr = fopen_s(&out,filename.c_str(),"wt");
		if (nerr) {
			cout << "file: " << filename << " could not be opened\n";
			return EXIT_FAILURE;
		}
	}
	else out = stdout;
	
	// single cycle simulation
	read_memory(in);
	fprintf(out,"<html>\n");
	simulate(out);
	//fprintf(out,"<h3>Registers</h3>\n");
	//show_registers(out);
	/*
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	*/
	
	// pipeline
	
	rewind(in);
	read_memory(in);
	fclose(in);
#define __PIPE
	pipeline(out);
	fprintf(out,"<h3>Registers</h3>\n");
	show_registers(out);
	fprintf(out,"<h3>Memory</h3>\n");
	show_data_memory(out);
	
	return EXIT_SUCCESS;
}
	

#if defined(__SIMULATE)

void simulate(FILE *out)
{
	int pc, nextpc;
	int inst, seim, im10, maddr;
	int opcode, rd=0, ra=0, rb=0;
	int qa, qb, qd=0;
	bool flag, done = false;
	if (out) {
		fprintf(out,"<h3>Simulation Results</h3>\n");
		fprintf(out,"<p><table border>\n");
		fprintf(out,"<tr bgcolor=CCCCFF><th>#<th>pc<th>op<th>rd<th>qd<th>addr\n");
	}
	nextpc = 0;
	while (!done) {
		inst = mem[nextpc];
		opcode = inst>>13;
		rd = (inst>>10)&0x7;
		ra = (inst>>7)&0x7;
		rb = inst&0x7;
		seim = inst&0x7F;
		im10 = inst&0x3ff;
		if (seim>>6) seim = (0xFFFFFF80)| seim;
		qa = reg(ra);
		qb = reg(rb);
		maddr = qa + seim;
		if (inst==HALT) break;
		pc = nextpc;
		nextpc++;
		switch (opcode) {
			case ADD:
				qd = qa + qb;
				break;
			case ADDI:
				qd = qa + seim;
				break;
			case NAND:
				qd = ~(qa & qb);
				qd &= 0xFFFF;
				break;
			case LUI:
				qd = (im10<<6);
				break;
			case SW: // SW
				qd = reg(rd);
				if (mem_check(maddr)) mem[maddr] = qd;
				break;
			case LW:
				if (mem_check(maddr)) qd = mem[maddr];
				break;
			case BNE:
				flag = reg(rd)!=qa;
				if (flag) {
					nextpc += seim;
					qd = nextpc;
				}
				else qd = 0;
				break;
			case JALR:
				qd = nextpc;
				nextpc = qa;
				break;
				
		}
		qd &= 0xFFFF;
		if (rd && opcode!=SW && opcode!=BNE) regs[rd] = qd;
		int ndx = nsteps;
		steps[ndx].set(pc,opcode,rd,qd);
		if (opcode==SW || opcode==LW) steps[ndx].addr = maddr;
		if (out) steps[ndx].show(out,ndx);
		if (++nsteps>=MAX_CYCLES) break;
	}
	if (out) fprintf(out,"</table>\n");
}

#endif

void showpc(FILE *out, int pc, bool color)
{
	if (color) fprintf(out,"<td bgcolor=E0E0E0");
	else fprintf(out,"<td");
	fprintf(out," >%04x",pc);
}

#if defined(__PIPE)

class iface {
public:
	bool active;
	bool color;
	bool halt;
	int ncycle;
	int instr;
	int pc;
};

class IFID: public iface {
public:
	int op, fn;
	int rd, ra, rb;
	void show(FILE *out);
	void bubble();
};

void IFID::show(FILE *out)
{
	char *str;
	if (active) {
		showpc(out,pc,color);
		str = op==0? funcname[fn]: opname[op];
		fprintf(out,"<br>%s",halt? "halt":str);
	}
	else fprintf(out,"<TD><BR>");
}

void IFID::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	op = 0;
	rd = 0;
}

class IDEX: public iface {
public:
	int control, muxc;
	int rd, ra, rb;
	int qa, qb;
	void show(FILE *out);
	void bubble();
	void setreg();
};

void IDEX::bubble()
{
	active = false;
	halt = false;
	instr = 0;
	rd = 0;
}

void IDEX::setreg()
{
	qa = reg(ra);
	qb = reg(rb);
}

void IDEX::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>\n");
		else fprintf(out,"<br>%d %d %d\n",rd,ra,rb);
	}
	else fprintf(out,"<td><br>");
}

class EXWB: public iface {
public:
	bool ma;
	int maddr;
	int jump;
	int rd;
	int qd;
	void show(FILE *out);
	void showmem(FILE *out);
};

void EXWB::showmem(FILE *out)
{
	int op = instr>>13;
	if (op==LW) fprintf(out,"qd = %04x from mem[0x%x]\n",qd,maddr);
	if (op==SW) fprintf(out,"mem[0x%x] = %04x\n",maddr,qd);
}

void EXWB::show(FILE *out)
{
	int op = instr>>13;
	bool flag = (op==LW || op==SW);
	if (active) {
		showpc(out,pc,color);
		if (halt) fprintf(out,"<br><br>");
		else fprintf(out,"<br>%04x",(flag? maddr: qd));
	}
	else fprintf(out,"<td><br>");
}

class WBIF: public iface {
public:
	int rd;
	int qd;
	void show(FILE *out);
};

void WBIF::show(FILE *out)
{
	if (active) {
		showpc(out,pc,color);
		fprintf(out,"<br>%04x",qd);
	}
	else fprintf(out,"<td><br>");
}


IFID IFID_old, IFID_new;
IDEX IDEX_old, IDEX_new;
EXWB EXWB_old, EXWB_new;
WBIF WBIF_old, WBIF_new;

#define NO_FORWARDING

void stage_IF(int nextpc, int ncycle, IFID &IFID_old, IFID &IFID_new) {
	mips_format_t ir;
	int op, fn;
	int ra, rb, rd;
	int range_error;
	// IF stage
	range_error = inst_memory(nextpc,ir.w);
	IFID_new.active = !IFID_old.halt;
	IFID_new.color = (ncycle&3)==2;
	IFID_new.ncycle = ncycle;
	IFID_new.instr = ir.w;
	IFID_new.halt = (ir.w==HALT) || IFID_old.halt;
	IFID_new.pc = nextpc;
	// obtain instruction fields
	op = ir.op;
	rd = ir.rd;
	ra = ir.rs;
	rb = ir.rt;
	fn = ir.fn;
	if (op>0) rd = rb;

	IFID_new.op = op;
	IFID_new.fn = fn;
	bool flag = op==SW || op==BNE;
	IFID_new.rd = flag? 0: rd;
	IFID_new.ra = ra;
	IFID_new.rb = rb;	
}

void stage_ID(IFID &IFID_old, IDEX &IDEX_new)
{
	int op, fn;
	int rcontrol, icontrol, control;
	int imux, muxc;
	op = IFID_old.op;
	fn = IFID_old.fn;
	rformat(fn,rcontrol);
	iformat(op,icontrol,imux);
	control = op==0? rcontrol: icontrol;
	muxc = op==0? 0: imux;

	IDEX_new.active = IFID_old.active;
	IDEX_new.color = IFID_old.color;
	IDEX_new.ncycle = IFID_old.ncycle;
	IDEX_new.instr = IFID_old.instr;
	IDEX_new.halt = IFID_old.halt;
	IDEX_new.pc = IFID_old.pc;
	IDEX_new.control = control;
	IDEX_new.muxc = muxc;
	IDEX_new.rd = IFID_old.rd;
	IDEX_new.ra = IFID_old.ra;
	IDEX_new.rb = IFID_old.rb;
	IDEX_new.setreg();
}

void stage_EX(IDEX &IDEX_old, EXWB &EXWB_old, EXWB &EXWB_new, FILE *out)
{
	mips_format_t ir;
	EXWB_new.active = IDEX_old.active;
	EXWB_new.color = IDEX_old.color;
	EXWB_new.halt = IDEX_old.halt;
	EXWB_new.ncycle = IDEX_old.ncycle;
	EXWB_new.pc = IDEX_old.pc;
	EXWB_new.instr = IDEX_old.instr;
	EXWB_new.rd = IDEX_old.rd;
	ir.w = IDEX_old.instr;
	int qa = reg(IDEX_old.ra);
	int qb = reg(IDEX_old.rb);
	// data forwarding
#define FORWARDING
#if defined(FORWARDING)
	int ncycle = IDEX_old.ncycle + 2;
	if (IDEX_old.ra == EXWB_old.rd && IDEX_old.ra !=0) {
		qa = EXWB_old.qd;
		fprintf(stderr,"(cycle %d) qa forwarded: %04x\n",ncycle,qa);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=5>qa forwarded: %04x\n",qa);
	}
	if (IDEX_old.rb == EXWB_old.rd && IDEX_old.rb !=0) {
		qb = EXWB_old.qd;
		fprintf(stderr,"(cycle %d) qb forwarded: %04x\n",ncycle,qb);
		fprintf(out,"<tr align=center><td><br>");
		fprintf(out,"<td bgcolor=FFFF00 colspan=5>qb forwarded: %04x\n",qb);
	}
#endif
	//

	int maddr = 0; //qa + seim;
	EXWB_new.maddr = maddr;
	int control = IDEX_old.control;
	int muxc = IDEX_old.muxc;
	//int jump = 0;
	int qt = mux1(qb, ir.immed, muxc);
	int qd = alu(control, qa, qt);		
	EXWB_new.ma = false;
	EXWB_new.qd = qd;
}

void stage_WB(EXWB &EXWB_old, WBIF &WBIF_new)
{
	WBIF_new.active = EXWB_old.active;
	WBIF_new.color = EXWB_old.color;
	WBIF_new.halt = EXWB_old.halt;
	WBIF_new.ncycle = EXWB_old.ncycle;
	WBIF_new.pc = EXWB_old.pc;
	WBIF_new.instr = EXWB_old.instr;
	int rd = EXWB_old.rd;
	int qd = EXWB_old.qd;
	reg_wb(rd,qd);
	WBIF_new.rd = rd;
	WBIF_new.qd = qd;
};


void pipeline(FILE *out)
{
	int nextpc, ncycle=0;
	int ndx = 0;
	bool done = false;
	if (out) {
		fprintf(out,"<h3>Pipeline Results</h3>\n");
		fprintf(out,"<p><table border>\n");
		fprintf(out,"<tr bgcolor=CCCCFF><th>#<th>IF<th>ID<th>EX<th>WB<th>ok\n");
	}
	nextpc = 0;
	ncycle = 0;
	while (!done) {
		// IF stage
		stage_IF(nextpc,ncycle,IFID_old,IFID_new);
		nextpc++;
		// ID stage
		stage_ID(IFID_old,IDEX_new);
		//  EX Stage
		stage_EX(IDEX_old,EXWB_old,EXWB_new,out);
		// WB Stage
		stage_WB(EXWB_old,WBIF_new);
		//
		if (++ncycle>=MAX_CYCLES) break;
		if (WBIF_new.instr==HALT) break;
		// show state
		fprintf(out,"<tr align=center><td>%d",ncycle);
		IFID_new.show(out);
		IDEX_new.show(out);
		EXWB_new.show(out);
		WBIF_new.show(out);
		if (WBIF_new.active) {
			int qd = steps[ndx].qd;
			int pc = steps[ndx].pc;
			bool flag = WBIF_new.qd==qd &&
				    WBIF_new.pc==pc;
			if (flag) fprintf(out,"<td>");
			else fprintf(out,"<td bgcolor=FF0000>");
			fprintf(out,"%04x<br>%04x\n",pc,qd);
			ndx++;
		}
		else fprintf(out,"<td><br>\n");
		if (EXWB_new.ma) {
			fprintf(out,"<tr align=center><td>%d",++ncycle);
			fprintf(out,"<td colspan=5 align=center>");
			EXWB_new.showmem(out);
		}
		// change state
		IFID_old = IFID_new;
		IDEX_old = IDEX_new;
		EXWB_old = EXWB_new;
		WBIF_old = WBIF_new;
	}
	if (out) fprintf(out,"</table>\n");
}
		


void parse(FILE *in, FILE *out)
{
	char buf[512];
	char *inp, *next;
	int opcode, ra, rb, rc, field;
	int count, code, n;
	string token;
	bool done = false;
	/*
	 * Go through input file, line by line
	 */
	n = 0;
	while (!done) {
		fgets(buf,511,in);
		if (feof(in)) break;
		//fprintf(out,"%s",buf);
		if (*buf=='#') continue;
		next = buf;
		/*
		 * parse current line
		 */
		count = 0;
		opcode = ra = rb = rc = 0;
		while (next) {
			/*
			 * search for '<', and copy prior text to
			 * output stream
 */
			inp = next;
			next = strqtrm(inp,',');
			sscanf_s(inp,"%d",&field);
			//fprintf(out,"count %d field %d\n",count,field);
			switch (count) {
			case 0:
				token = inp;
				//fprintf(out,"%s %d\n",token.c_str(),opcode);
				count = 1;
				break;
			case 1:
				ra = field;
				count = 2;
				break;
			case 2:
				rb = field;
				count = 3;
				break;
			case 3:
				rc = field;
				count = 4;
				break;
			default:
				break;
			}
		}
		code = ((opcode&0x7)<<13) | ((ra&0x7)<<10) | ((rb&0x7)<<7) | (rc&0x7F);;
		fprintf(out,"%04X\n",code);
		n++;
#if defined(STUFF)
			/*
			 * next points to character after ',' (if ','
			 * was found).
			 */
			/*
			if (next) {
				/*
				*  isolate (in html) text up to the character '>'
				*/
				html = next;
				next = strqtrm(html,'>');
				/*
				*  check for known codes
				*/
				string token = html;
				char *value = strqtrm(html,' ');
				if (token.substr(0,7).compare("include")==0) {
					include(out,value);
					continue; // go to next line
				}
				else if (token.substr(0,4).compare("copy")==0) {
					copy(out,value);
					continue; // go to next line
				}
				else if (token.substr(0,6).compare("insert")==0) {
					parse_insert(value,out);
				}
				/*
				 *  otherwise just echo the html code
				 */
				else fprintf(out,"<%s>",token.c_str());
			}
#endif
	}
	while (n++ < 32) fprintf(out,"0000\n");
}

#endif
