/* \file drisc.cpp
 *
 *  RISC CPU Implementation
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream>
#include <string>
using namespace std;
#include "strtoken.h"
#include "mips.h"
#include "mips_table.h"
#include "opcodes.h"
#include "drisc.h"

// category 1
int inst[256];
int kwords = 0;

// category 2
int data[512];
int nwords = 512;
int mwords = 0;

cycle steps[MAX_CYCLES];
int nc;

void init_regs();

int inst_memory(int addr, int &q)
{
	if (addr<0 || addr>=kwords) return 1;
	q = inst[addr];
	return 0;
}

int data_memory(int addr, int qd, bool write_enable, int &q)
{
	if (addr<0 || addr>=nwords) return 1;
	q = data[addr];
	if (write_enable) data[addr] = qd;
	return 0;
}
	

void show_memory(FILE *out, int mem[], int n)
{
	int i,flag=0;
	fprintf(out,"<p><table border>\n");
	fprintf(out,"<p><tr bgcolor=CCCCFF><th>addr");
	for (i=0; i<16; i++) fprintf(out,"<th>%X",i);
	fprintf(out,"\n");
	for (i=0; i<n; i++) {
		flag = i%16;
		if (flag==0) fprintf(out,"<tr align=center><td bgcolor=FFFFCC>%08X",i);
		fprintf(out,"<td>%08X",mem[i]);
		if (flag==7) fprintf(out,"\n");
	}
	while (flag++ < 15) fprintf(out,"<td><br>");
	fprintf(out,"</table>\n");
	
}

void show_data_memory(FILE *out)
{
	show_memory(out,data,32);
}

void show_inst_memory(FILE *out)
{
	show_memory(out,inst,kwords);
}

int read_memory(FILE *in)
{
	char buf[512];
	char *inp, *next;
	int field, n;
	int category =0;
	bool done = false;
	n = 0;
	init_regs();
	// initialize data memory
	for (n=0; n<nwords; n++) data[n] = 0;
	while (!done) {
		fgets(buf,511,in);
		if (feof(in)) break;
		next = buf;
		while (*next==' ') next++;
		if (isalpha(*next)) {
			inp = strtoken(next,'\n'); 
			if (strcmp(inp,"inst")==0) {
				if (category==2) mwords = n;
				category = 1;
				n = 0;
				continue;
			}
			else if (strcmp(inp,"data")==0) {
				if (category==1) kwords = n;
				category = 2;
				n = 0;
				continue;
			}
			else if (strcmp(inp,"end")==0) {
				if (category==1) kwords = n;
				else if (category==2) mwords = n;
				break;
			}
		}
		while (next) {
			inp = next;
			while (*inp==' ') inp++; // eat blanks
			if (*inp=='\n') break;
			next = strqtrm(inp,' ');
			sscanf_s(inp,"%x",&field);
			//if (field>0) printf("%04X\n",field);
			if (category==2) data[n++] = field;
			else if (category==1) inst[n++] = field;
		}
	}
	return kwords;
}


int regs[32];


void init_regs()
{
	int i;
	for (i=0; i<32; i++) regs[i] = i;
	regs[0] = 0xBAD;
}

int reg(int n) {
	if (n>0) return regs[n];
	return 0;
}

void reg_wb(int rd, int qd)
{
	if (rd>0) regs[rd] = qd;
}


void show_registers(FILE *out)
{
	int i, group;
	int ht=10;
	fprintf(out,"<p><table border>\n");
	for (group=0; group<4; group++) {
		fprintf(out,"<p><tr bgcolor=CCCCFF><th>format");
		for (i=0; i<8; i++) fprintf(out,"<th>r%d",i+8*group);
		fprintf(out,"\n");
		fprintf(out,"<tr align=center><th>hex");
		for (i=0; i<8; i++) {
			fprintf(out,"<td>%08X",reg(i+8*group));
		}
		fprintf(out,"</tr>\n");
		fprintf(out,"<tr align=center><th>decimal");
		for (i=0; i<8; i++) {
			fprintf(out,"<td>%d",reg(i+8*group));
		}
		fprintf(out,"</tr>\n");
		if (group==3) break;
		fprintf(out,"<tr><td colspan=9 style=\"height:%dpx\"></tr>\n",ht);
	}
	fprintf(out,"</table>\n");
}

void rformat(int fn, int &control)
{
	switch (fn) {
	case ADD:
		control = 0;
		break;
	case ADDU:
		control = 2;
		break;
	case SUB:
		control = 1;
		break;
	case SUBU:
		control = 3;
		break;
	case AND:
		control = 4;
		break;
	case OR:
		control = 5;
		break;
	case NOR:
		control = 6;
		break;
	case XOR:
		control = 7;
		break;
	default:
		control = -1;
		break;
	}
}

int mux1(int qb, int immed, int muxi)
{
	int qd, immedse;
	if (immed>>15) immedse = 0xFFFF0000| immed&0xFFFF;
	else immedse = immed;
	switch (muxi) {
	case 0:
		qd = qb;
		break;
	case 1:
		qd = immedse;
		break;
	case 2:
		qd = immed&0xFFFF;
		break;
	case 3:
		qd = immed<<16;
		break;
	default:
		qd = 0;
	}
	return qd;
}
	

void iformat(int op, int &control, int &muxi)
{
	
	switch (op) {
	case ADDI:
		control=0;
		muxi = 1;
		break;
	case ADDIU:
		control=2;
		muxi = 1;
		break;
	case ANDI:
		control=4;
		muxi = 2;
		break;
	case ORI:
		control=5;
		muxi = 2;
		break;
	case LUI:
		control=5;
		muxi = 3;
		break;
	case XORI:
		control=6;
		muxi = 2;
		break;
	case LW:
		control = LW_C<<4;
		muxi = 1;
		break;
	case SW:
		control = SW_C<<4;
		muxi = 1;
		break;
	default:
		control = 0xFFFF0000;
		muxi = 0;
	}
}

int alu(int control, int qa, int qb)
{
	int qd;
	switch (control) {
	case 0:
		qd = qa + qb;
		break;
	case 1:
		qd = qa - qb;
		break;
	case 2:
		qd = (unsigned) qa + (unsigned) qb;
		break;
	case 3:
		qd = (unsigned) qa + (unsigned) qb;
		break;
	case 4:
		qd = qa & qb;
		break;
	case 5:
		qd = qa | qb;
		break;
	case 6:
		qd = ~(qa | qb);
		break;
	case 7:
		qd = qa ^ qb;
		break;
	default:
		qd = 0;
		break;
	}
	return qd;
}




void cycle::set(int pci, int opi, int fni, int rdi, int qdi)
{
	pc = pci;
	op = opi;
	fn = fni;
	rd = rdi;
	qd = qdi;
}

void cycle::show(FILE *out, int ncycle)
{
	fprintf(out,"<tr><td bgcolor=FFFFCC>%d<td>%02X",ncycle+1,pc);
	char *str = op==0? funcname[fn]: opname[op];
	if (halt) fprintf(out,"<td colspan=5 align=center>halt</tr>\n");
	else {
		fprintf(out,"<td>%s<td>%d<td>%04X<td align=right>%d<td>",str,rd,qd,qd);
		if (op==LW || op==SW) fprintf(out,"%04X\n",addr);
		else fprintf(out,"<br>\n");
	}
}


void simulate(FILE *out)
{
	mips_format_t ir;
	int pc, nextpc;
	int immed,  maddr;
	int range_error;
	int op, fn, rd=0, ra=0, rb=0;
	int qa, qb, qc=0, qd=0;
	int qt;
	int control, rcontrol, icontrol, imux, muxc;
	bool done = false;
	fprintf(out,"<h3>Simulation Results</h3>\n");
	fprintf(out,"<p><table border>\n");
	fprintf(out,"<tr bgcolor=CCCCFF><th>#<th>pc<th>op<th>rd<th>qd (hex)<th>qd (dec)<th>addr\n");
	nextpc = 0;
	nc = 0;
	while (!done) {
		range_error = inst_memory(nextpc, ir.w);
		if (range_error) {
			fprintf(stderr,"instruction memory error for address %d\n",nextpc);
			break;
		}
		// decode instruction
		op = ir.op;
		rd = ir.rd;
		ra = ir.rs;
		rb = ir.rt;
		fn = ir.fn;
		immed = ir.immed;
		if (op>0) rd = rb;
		// read registers
		qa = reg(ra);
		qb = reg(rb);
		pc = nextpc;
		nextpc++;
		// decode instruction
		rformat(fn,rcontrol);
		iformat(op,icontrol,imux);
		control = op==0? rcontrol: icontrol;
		muxc = op==0? 0: imux;
		// execute instruction
		qt = mux1(qb, immed, muxc);
		qc = alu(control&0x7, qa, qt);
		// data memory access
		int qm=0;
		int opc = (control>>4)&0xF;
		maddr = qc;
		if (opc==SW_C || opc==LW_C) {	  
			range_error =  data_memory(qc, qb, opc==SW_C, qm);
			if (range_error) {
				fprintf(stderr,"data memory error for address %d\n",qc);
				break;
			}
		}
		qd = (opc==LW_C? qm: qc);
		// write back
		if (opc!=SW_C) reg_wb(rd,qd);
		// save for later use
		bool halt = ir.w==HALT;
		steps[nc].set(pc,op,fn,rd,(opc==SW_C? qb: qd));
		steps[nc].halt = halt;
		if (opc==SW_C || opc==LW_C) steps[nc].addr = maddr;
		if (out) steps[nc].show(out,nc);
		nc++;
		if (halt) break;
		/*
		if (op>0) {
			fprintf(out,"<td>%08X\n",immed);
			if (range_error) printf("address %x out of range\n",maddr);
		}
		else fprintf(out,"<td><br>\n");
		*/

		if (nc>MAX_CYCLES) break;
	}
	fprintf(out,"</table>\n");
}


#if defined(__MAIN)

int main(int argc, char *argv[])
{
	FILE *in, *out;
	string filename;
	errno_t nerr;
	if (argc<2) {
		cout << "usage: drisc mem.txt [output]\n";
		return EXIT_FAILURE;
	}
	filename = argv[1];
	nerr = fopen_s(&in,filename.c_str(),"r");
	if (nerr) {
		cout << "file: " << filename << " not found\n";
		return EXIT_FAILURE;	}
	read_memory(in);
	fclose(in);
	if (argc<3) out = stdout;
	else {
		filename = argv[2];
		nerr = fopen_s(&out,filename.c_str(),"wt");
		if (nerr) {
			cout << "file: " << filename << "not opened\n";
			return EXIT_FAILURE;
		}
	}
	fprintf(out,"<html>\n");
	//fprintf(out,"<h3>Instruction Memory</h3>\n");
	//show_inst_memory(out);

	simulate(out);

	fprintf(out,"<h3>Registers</h3>\n");
	show_registers(out);

	fprintf(out,"<h3>Data Memory</h3>\n");
	show_data_memory(out);

	fprintf(out,"</html>\n");
	return EXIT_SUCCESS;
}
#endif
