/* \file drisc.cpp
 *
 *  Simulator
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream>
#include <string>
using namespace std;
#include "strtoken.h"
#include "cpu.h"

void simulate(FILE *out);

int main(int argc, char *argv[])
{
	FILE *in, *out;
	string filename;
	errno_t nerr;
	if (argc<2) {
		cout << "usage: drisc mem.txt [output_html]\n";
		return EXIT_FAILURE;
	}
	filename = argv[1];
	nerr = fopen_s(&in,filename.c_str(),"r");
	if (nerr) {
		cout << "file: " << filename << " not found\n";
		return EXIT_FAILURE;	}
	read_memory(in);
	fclose(in);
	if (argc<3) out = stdout;
	else {
		filename = argv[2];
		nerr = fopen_s(&out,filename.c_str(),"wt");
		if (nerr) {
			cout << "file: " << filename << "not opened\n";
			return EXIT_FAILURE;
		}
	}
	fprintf(out,"<html>\n");
	fprintf(out,"<h3>Instructions</h3>\n");
	show_inst_memory(out);

	simulate(out);

	fprintf(out,"<h3>Registers</h3>\n");
	show_registers(out);
/*
	fprintf(out,"<h3>Memory</h3>\n");
	show_memory(out);
*/
	fprintf(out,"</html>\n");
	return EXIT_SUCCESS;
}

#if defined(__EXTRA__)

int mem_check(int addr, int nwords)
{
	if (addr<0 || addr>=nwords) {
		return 1;
	}
	return 0;
}

void show_memory(FILE *out, int mem[], int n)
{
	int i,flag=0;
	fprintf(out,"<p><table border>\n");
	fprintf(out,"<p><tr bgcolor=CCCCFF><th>addr");
	for (i=0; i<16; i++) fprintf(out,"<th>%X",i);
	fprintf(out,"\n");
	for (i=0; i<n; i++) {
		flag = i%16;
		if (flag==0) fprintf(out,"<tr align=center><td bgcolor=FFFFCC>%08X",i);
		fprintf(out,"<td>%08X",mem[i]);
		if (flag==7) fprintf(out,"\n");
	}
	while (flag++ < 15) fprintf(out,"<td><br>");
	fprintf(out,"</table>\n");
	
}

int read_memory(FILE *in, int mem[])
{
	char buf[512];
	char *inp, *next;
	int field, n;
	bool done = false;
	n = 0;
	while (!done) {
		fgets(buf,511,in);
		if (feof(in)) break;
		next = buf;
		while (*next==' ') next++;
		while (next) {
			inp = next;
			while (*inp==' ') inp++; // eat blanks
			if (*inp=='\n') break;
			next = strqtrm(inp,' ');
			sscanf_s(inp,"%x",&field);
			//if (field>0) printf("%04X\n",field);
			mem[n++] = field;
		}
	}
	return n;
}


int regs[] = {
	0x0BAD,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
	16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31
	};

int reg(int n) {
	if (n>0) return regs[n];
	return 0;
}

void reg_wb(int rd, int qd)
{
	if (rd>0) regs[rd] = qd;
}


void show_registers(FILE *out)
{
	int i, group;
	fprintf(out,"<p><table border>\n");
	for (group=0; group<4; group++) {
		fprintf(out,"<p><tr bgcolor=CCCCFF><th>%s",group? "<br>":"format");
		for (i=1; i<8; i++) fprintf(out,"<th>r%d",i+8*group);
		fprintf(out,"\n");
		fprintf(out,"<tr align=center><th>hex");
		for (i=1; i<8; i++) {
			fprintf(out,"<td>%08X",reg(i+8*group));
		}
		fprintf(out,"</tr>\n");
		fprintf(out,"<tr align=center><th>decimal");
		for (i=1; i<8; i++) {
			fprintf(out,"<td>%d",reg(i+8*group));
		}
		fprintf(out,"</tr>\n");
	}
	fprintf(out,"</table>\n");
}



const int HALT = 0x0000;

int rformat(int fn, int qa, int qb)
{
	int qd;
	switch (fn) {
	case ADD:
		qd = qa + qb;
		break;
	case ADDU:
		qd = qa + qb;  // no distinction for unsigned addition
		break;
	case SUB:
		qd = qa - qb;
		break;
	case SUBU:
		qd = qa - qb;  // no distinction for unsigned subtraction
		break;
	case AND:
		qd = qa & qb;
		break;
	case OR:
		qd = qa | qb;
		break;
	case NOR:
		qd = ~(qa | qb);
		break;
	case XOR:
		qd = qa ^ qb;
		break;
	default:
		qd = 0;
		break;
	}
	return qd;
}

int iformat(int op, int ra, int immed)
{
	int qd, immedse;
	if (immed>>15) immedse = 0xFFFF0000| immed&0xFFFF;
	else immedse = immed;
	
	switch (op) {
	case ADDI:
		qd = ra + immedse;
		break;
	case ADDIU:
		qd = ra + immedse;
		break;
	case ANDI:
		qd = ra & immed;
		break;
	case ORI:
		qd = ra | immed;
		break;
	case LUI:
		qd = immed<<16;
		break;
	case XORI:
		qd = ra ^ immed;
		break;
	default:
		qd = 0;
	}
	return qd;
}

#endif

		


void parse(FILE *in, FILE *out)
{
	char buf[512];
	char *inp, *next;
	int opcode, ra, rb, rc, field;
	int count, code, n;
	string token;
	bool done = false;
	/*
	 * Go through input file, line by line
	 */
	n = 0;
	while (!done) {
		fgets(buf,511,in);
		if (feof(in)) break;
		//fprintf(out,"%s",buf);
		if (*buf=='#') continue;
		next = buf;
		/*
		 * parse current line
		 */
		count = 0;
		opcode = ra = rb = rc = 0;
		while (next) {
			/*
			 * search for '<', and copy prior text to
			 * output stream
 */
			inp = next;
			next = strqtrm(inp,',');
			sscanf_s(inp,"%d",&field);
			//fprintf(out,"count %d field %d\n",count,field);
			switch (count) {
			case 0:
				token = inp;
				//fprintf(out,"%s %d\n",token.c_str(),opcode);
				count = 1;
				break;
			case 1:
				ra = field;
				count = 2;
				break;
			case 2:
				rb = field;
				count = 3;
				break;
			case 3:
				rc = field;
				count = 4;
				break;
			default:
				break;
			}
		}
		code = ((opcode&0x7)<<13) | ((ra&0x7)<<10) | ((rb&0x7)<<7) | (rc&0x7F);;
		fprintf(out,"%04X\n",code);
		n++;
	}
	while (n++ < 32) fprintf(out,"0000\n");
}
